'use client'

import { useState, useMemo } from 'react'
import { useRouter, useParams } from 'next/navigation'
import Link from 'next/link'
import {
    ArrowLeft, MapPin, Phone, MessageCircle, Calendar,
    Edit3, X, Check, ChevronRight, Navigation, Clock,
    FileText, AlertTriangle, Loader2, Trash2
} from 'lucide-react'
import AnimatedPage, { StaggeredItem } from '@/components/AnimatedPage'
import { useCRMStore } from '@/store/crmStore'
import { useToast } from '@/components/Toast'
import { getStageLabel, getStageColor, PIPELINE_STAGES, ACTIVITY_TYPES, ACTIVITY_RESULTS, LOST_REASONS } from '@/lib/utils'

const activityIcons: Record<string, any> = {
    visit: MapPin, phone_call: Phone, wa_chat: MessageCircle,
    survey_lokasi: Navigation, follow_up: Clock, presentasi: FileText,
    negosiasi: FileText, registrasi_psb: Check,
}

export default function LeadDetailPage() {
    const router = useRouter()
    const params = useParams()
    const leadId = params.id as string

    const lead = useCRMStore(s => s.leads.find(l => l.id === leadId))
    const updateLead = useCRMStore(s => s.updateLead)
    const deleteLead = useCRMStore(s => s.deleteLead)
    const moveLead = useCRMStore(s => s.moveLead)
    const addActivity = useCRMStore(s => s.addActivity)
    const allActivities = useCRMStore(s => s.activities)
    const activities = useMemo(() => allActivities.filter(a => a.leadId === leadId), [allActivities, leadId])
    const { addToast } = useToast()

    const [showLogActivity, setShowLogActivity] = useState(false)
    const [showStageChange, setShowStageChange] = useState(false)
    const [showLostModal, setShowLostModal] = useState(false)
    const [editing, setEditing] = useState(false)

    // Activity form
    const [activityForm, setActivityForm] = useState({
        activityType: 'visit', result: 'interested', notes: '',
    })

    // Lost form
    const [lostReason, setLostReason] = useState('')
    const [lostNotes, setLostNotes] = useState('')

    if (!lead) {
        return (
            <AnimatedPage className="max-w-2xl mx-auto" style={{ paddingTop: '4rem', textAlign: 'center' }}>
                <AlertTriangle size={48} style={{ color: 'var(--color-warning)', margin: '0 auto 1rem' }} />
                <h2 style={{ fontSize: 'var(--text-xl)', fontWeight: 700 }}>Lead Tidak Ditemukan</h2>
                <p style={{ fontSize: 'var(--text-sm)', color: 'var(--color-muted)', marginTop: 4 }}>Lead dengan ID ini tidak ada di database.</p>
                <Link href="/leads" className="btn-accent" style={{ marginTop: '1.5rem', display: 'inline-flex' }}>
                    ← Kembali ke Leads
                </Link>
            </AnimatedPage>
        )
    }

    const handleLogActivity = () => {
        const userName = localStorage.getItem('crm-user-name') || 'Admin'
        addActivity({
            leadId: lead.id,
            leadName: lead.namaLengkap,
            activityType: activityForm.activityType,
            activityDate: new Date().toISOString(),
            result: activityForm.result,
            notes: activityForm.notes,
            sales: userName,
            area: lead.kecamatan,
        })
        setShowLogActivity(false)
        setActivityForm({ activityType: 'visit', result: 'interested', notes: '' })
        addToast('success', 'Aktivitas dicatat!', `${activityForm.activityType} untuk ${lead.namaLengkap}`)
    }

    const handleStageUpdate = (newStage: string) => {
        moveLead(lead.id, newStage)
        setShowStageChange(false)
        addToast('success', 'Stage diperbarui', `${lead.namaLengkap} → ${getStageLabel(newStage)}`)
    }

    const handleMarkLost = () => {
        updateLead(lead.id, { pipelineStage: 'lost', lostReason, lostNotes })
        setShowLostModal(false)
        addToast('info', 'Lead ditandai Lost', lead.namaLengkap)
    }

    const handleDelete = () => {
        if (confirm(`Hapus lead ${lead.namaLengkap}? Tindakan ini tidak dapat dibatalkan.`)) {
            deleteLead(lead.id)
            addToast('info', 'Lead dihapus', lead.namaLengkap)
            router.push('/leads')
        }
    }

    const daysInStage = Math.floor((Date.now() - new Date(lead.pipelineStageDate).getTime()) / 86400000)

    return (
        <AnimatedPage className="max-w-2xl mx-auto space-y-4">
            {/* Header */}
            <div className="flex items-center gap-3">
                <Link href="/leads" style={{ color: 'var(--color-muted)', display: 'flex' }}>
                    <ArrowLeft size={20} />
                </Link>
                <div style={{ flex: 1 }}>
                    <h1 style={{ fontSize: 'var(--text-2xl)', fontWeight: 700 }}>{lead.namaLengkap}</h1>
                    <div className="flex items-center gap-2" style={{ marginTop: 2 }}>
                        <span style={{
                            fontSize: 'var(--text-xs)', fontWeight: 600, padding: '2px 8px',
                            borderRadius: 'var(--radius-full)', color: getStageColor(lead.pipelineStage),
                            background: getStageColor(lead.pipelineStage) + '18',
                        }}>
                            {getStageLabel(lead.pipelineStage)}
                        </span>
                        <span style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>{daysInStage} hari di stage ini</span>
                    </div>
                </div>
                <button onClick={handleDelete} style={{ color: 'var(--color-danger)', background: 'none', border: 'none', cursor: 'pointer', padding: 4 }}>
                    <Trash2 size={16} />
                </button>
            </div>

            {/* Action Buttons */}
            <StaggeredItem index={1}>
                <div className="flex flex-wrap gap-2">
                    <button onClick={() => setShowLogActivity(true)} className="btn-accent" style={{ flex: '1 0 auto' }}>
                        <FileText size={14} /> Log Aktivitas
                    </button>
                    <button onClick={() => setShowStageChange(true)} className="btn-outline" style={{ flex: '1 0 auto' }}>
                        <ChevronRight size={14} /> Ubah Stage
                    </button>
                    {lead.pipelineStage !== 'lost' && lead.pipelineStage !== 'won' && (
                        <button onClick={() => setShowLostModal(true)} className="btn-outline" style={{ flex: '1 0 auto', color: 'var(--color-danger)', borderColor: 'var(--color-danger)' }}>
                            <X size={14} /> Mark Lost
                        </button>
                    )}
                    <a href={`https://wa.me/${lead.noWa}`} target="_blank" className="btn-outline" style={{ flex: '1 0 auto', color: '#10B981', borderColor: '#10B981' }}>
                        <MessageCircle size={14} /> WhatsApp
                    </a>
                </div>
            </StaggeredItem>

            {/* Contact Info */}
            <StaggeredItem index={2}>
                <div className="card" style={{ padding: '1.25rem' }}>
                    <h3 style={{ fontSize: 'var(--text-base)', fontWeight: 700, marginBottom: '0.75rem' }}>Informasi Kontak</h3>
                    <div className="grid grid-cols-1 sm:grid-cols-2 gap-3" style={{ fontSize: 'var(--text-sm)' }}>
                        <div className="flex items-center gap-2">
                            <Phone size={14} style={{ color: 'var(--color-muted)' }} />
                            <span>{lead.noWa}</span>
                        </div>
                        <div className="flex items-center gap-2">
                            <MapPin size={14} style={{ color: 'var(--color-muted)' }} />
                            <span>{lead.kecamatan}{lead.kelurahan ? `, ${lead.kelurahan}` : ''}</span>
                        </div>
                        {lead.alamat && (
                            <div className="sm:col-span-2 flex items-center gap-2">
                                <Navigation size={14} style={{ color: 'var(--color-muted)' }} />
                                <span>{lead.alamat}</span>
                            </div>
                        )}
                        {lead.paketInterest && (
                            <div className="flex items-center gap-2">
                                <span style={{ color: 'var(--color-muted)' }}>Paket:</span>
                                <span style={{ fontWeight: 600 }}>{lead.paketInterest}</span>
                            </div>
                        )}
                        {lead.estimatedValue && (
                            <div className="flex items-center gap-2">
                                <span style={{ color: 'var(--color-muted)' }}>Estimasi:</span>
                                <span style={{ fontWeight: 600 }}>Rp {lead.estimatedValue.toLocaleString('id-ID')}</span>
                            </div>
                        )}
                        {lead.notes && (
                            <div className="sm:col-span-2" style={{ padding: '0.75rem', background: 'var(--color-subtle)', borderRadius: 'var(--radius-md)', fontSize: 'var(--text-xs)', color: 'var(--color-muted)', lineHeight: 1.5 }}>
                                {lead.notes}
                            </div>
                        )}
                    </div>
                </div>
            </StaggeredItem>

            {/* Activity Timeline */}
            <StaggeredItem index={3}>
                <div className="card" style={{ padding: '1.25rem' }}>
                    <h3 style={{ fontSize: 'var(--text-base)', fontWeight: 700, marginBottom: '0.75rem' }}>Riwayat Aktivitas</h3>
                    {activities.length === 0 ? (
                        <div style={{ padding: '2rem', textAlign: 'center' }}>
                            <div className="empty-icon-pulse" style={{ display: 'inline-block' }}>
                                <Clock size={28} style={{ color: 'var(--color-muted)' }} />
                            </div>
                            <p style={{ fontSize: 'var(--text-sm)', color: 'var(--color-muted)', marginTop: '0.5rem' }}>
                                Belum ada aktivitas. Klik "Log Aktivitas" untuk mencatat.
                            </p>
                        </div>
                    ) : (
                        <div className="space-y-3">
                            {activities.map((act, i) => {
                                const ActIcon = activityIcons[act.activityType] || Clock
                                return (
                                    <div key={act.id} className="timeline-item">
                                        <div className="timeline-dot" style={{ borderColor: 'var(--color-accent-500)', background: 'var(--color-accent-100)' }}>
                                            <ActIcon size={10} style={{ color: 'var(--color-accent-500)' }} />
                                        </div>
                                        <div>
                                            <p style={{ fontSize: 'var(--text-sm)', fontWeight: 600 }}>
                                                {ACTIVITY_TYPES.find(t => t.value === act.activityType)?.label || act.activityType}
                                            </p>
                                            <p style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>
                                                {act.sales} · {new Date(act.activityDate).toLocaleDateString('id-ID')} · Hasil: {ACTIVITY_RESULTS.find(r => r.value === act.result)?.label || act.result}
                                            </p>
                                            {act.notes && <p style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)', marginTop: 2, fontStyle: 'italic' }}>{act.notes}</p>}
                                        </div>
                                    </div>
                                )
                            })}
                        </div>
                    )}
                </div>
            </StaggeredItem>

            {/* ── MODALS ──────────────────────────────── */}
            {/* Log Activity Modal */}
            {showLogActivity && (
                <div className="fixed inset-0 z-50 flex items-center justify-center p-4" style={{ background: 'rgba(0,0,0,0.4)' }}>
                    <div className="card card-enter" style={{ width: '100%', maxWidth: 420, padding: '1.5rem' }}>
                        <div className="flex items-center justify-between mb-4">
                            <h3 style={{ fontWeight: 700 }}>Log Aktivitas</h3>
                            <button onClick={() => setShowLogActivity(false)} style={{ background: 'none', border: 'none', cursor: 'pointer' }}><X size={18} /></button>
                        </div>
                        <div className="space-y-3">
                            <div>
                                <label className="form-label">Tipe Aktivitas</label>
                                <select className="form-input form-select" value={activityForm.activityType} onChange={e => setActivityForm(p => ({ ...p, activityType: e.target.value }))}>
                                    {ACTIVITY_TYPES.map(t => <option key={t.value} value={t.value}>{t.label}</option>)}
                                </select>
                            </div>
                            <div>
                                <label className="form-label">Hasil</label>
                                <select className="form-input form-select" value={activityForm.result} onChange={e => setActivityForm(p => ({ ...p, result: e.target.value }))}>
                                    {ACTIVITY_RESULTS.map(r => <option key={r.value} value={r.value}>{r.label}</option>)}
                                </select>
                            </div>
                            <div>
                                <label className="form-label">Catatan</label>
                                <textarea className="form-input" rows={3} value={activityForm.notes} onChange={e => setActivityForm(p => ({ ...p, notes: e.target.value }))} />
                            </div>
                            <button onClick={handleLogActivity} className="btn-accent" style={{ width: '100%' }}>
                                <Check size={14} /> Simpan
                            </button>
                        </div>
                    </div>
                </div>
            )}

            {/* Stage Change Modal */}
            {showStageChange && (
                <div className="fixed inset-0 z-50 flex items-center justify-center p-4" style={{ background: 'rgba(0,0,0,0.4)' }}>
                    <div className="card card-enter" style={{ width: '100%', maxWidth: 360, padding: '1.5rem' }}>
                        <div className="flex items-center justify-between mb-4">
                            <h3 style={{ fontWeight: 700 }}>Pindah Stage</h3>
                            <button onClick={() => setShowStageChange(false)} style={{ background: 'none', border: 'none', cursor: 'pointer' }}><X size={18} /></button>
                        </div>
                        <div className="space-y-2">
                            {PIPELINE_STAGES.filter(s => s.value !== lead.pipelineStage).map(s => (
                                <button key={s.value} onClick={() => handleStageUpdate(s.value)}
                                    style={{
                                        width: '100%', padding: '0.75rem', display: 'flex', alignItems: 'center', gap: 8,
                                        background: 'var(--color-subtle)', border: '1px solid var(--color-border)',
                                        borderRadius: 'var(--radius-md)', cursor: 'pointer', fontSize: 'var(--text-sm)', fontWeight: 600,
                                    }}>
                                    <div style={{ width: 8, height: 8, borderRadius: '50%', background: s.color }} />
                                    {s.label}
                                </button>
                            ))}
                        </div>
                    </div>
                </div>
            )}

            {/* Lost Modal */}
            {showLostModal && (
                <div className="fixed inset-0 z-50 flex items-center justify-center p-4" style={{ background: 'rgba(0,0,0,0.4)' }}>
                    <div className="card card-enter" style={{ width: '100%', maxWidth: 400, padding: '1.5rem' }}>
                        <div className="flex items-center justify-between mb-4">
                            <h3 style={{ fontWeight: 700 }}>Mark as Lost</h3>
                            <button onClick={() => setShowLostModal(false)} style={{ background: 'none', border: 'none', cursor: 'pointer' }}><X size={18} /></button>
                        </div>
                        <div className="space-y-3">
                            <div>
                                <label className="form-label">Alasan</label>
                                <select className="form-input form-select" value={lostReason} onChange={e => setLostReason(e.target.value)}>
                                    <option value="">Pilih alasan</option>
                                    {LOST_REASONS.map(r => <option key={r.value} value={r.value}>{r.label}</option>)}
                                </select>
                            </div>
                            <div>
                                <label className="form-label">Catatan</label>
                                <textarea className="form-input" rows={2} value={lostNotes} onChange={e => setLostNotes(e.target.value)} />
                            </div>
                            <button onClick={handleMarkLost} className="btn-accent" style={{ width: '100%', background: 'var(--color-danger)' }}>
                                Mark Lost
                            </button>
                        </div>
                    </div>
                </div>
            )}
        </AnimatedPage>
    )
}
